<?php
// inc/functions.php - Helper Functions

require_once __DIR__ . '/db.php';

/**
 * Log user activity
 */
function log_activity($user_id, $action, $details = '') {
    $sql = "INSERT INTO activity_logs (user_id, action, details, created_at) VALUES (?, ?, ?, NOW())";
    db_query($sql, [$user_id, $action, $details]);
}

/**
 * Get user info, create if not exists
 */
function get_or_create_user($user_id, $username, $first_name, $last_name = '') {
    $sql = "SELECT * FROM users WHERE user_id = ?";
    $user = db_query($sql, [$user_id])->fetch();
    
    if (!$user) {
        $sql = "INSERT INTO users (user_id, username, first_name, last_name, created_at) VALUES (?, ?, ?, ?, NOW())";
        db_query($sql, [$user_id, $username, $first_name, $last_name]);
        
        $sql = "SELECT * FROM users WHERE user_id = ?";
        $user = db_query($sql, [$user_id])->fetch();
        
        log_activity($user_id, 'user_created', "Username: $username");
    }
    
    return $user;
}

/**
 * Get all files
 */
function get_all_files() {
    $sql = "SELECT * FROM files ORDER BY created_at DESC";
    return db_query($sql)->fetchAll();
}

/**
 * Get file by ID
 */
function get_file_by_id($file_id) {
    $sql = "SELECT * FROM files WHERE id = ?";
    return db_query($sql, [$file_id])->fetch();
}

/**
 * Get file by public ID
 */
function get_file_by_public_id($public_id) {
    $sql = "SELECT * FROM files WHERE public_id = ?";
    return db_query($sql, [$public_id])->fetch();
}

/**
 * Check if user has joined required channels
 */
function check_channel_membership($user_id, $required_channels) {
    if (empty($required_channels)) return true;
    
    $channels = explode(',', $required_channels);
    $not_joined = [];
    
    foreach ($channels as $channel) {
        $channel = trim($channel);
        if (!empty($channel) && !is_telegram_member($user_id, $channel)) {
            $not_joined[] = $channel;
        }
    }
    
    return empty($not_joined) ? true : $not_joined;
}

/**
 * Create unlock request
 */
function create_unlock_request($user_id, $file_id) {
    // Check if request already exists
    $sql = "SELECT id FROM unlock_requests WHERE user_id = ? AND file_id = ? AND status = 'pending'";
    $existing = db_query($sql, [$user_id, $file_id])->fetch();
    
    if ($existing) {
        return $existing['id'];
    }
    
    $sql = "INSERT INTO unlock_requests (user_id, file_id, status, created_at) VALUES (?, ?, 'pending', NOW())";
    db_query($sql, [$user_id, $file_id]);
    
    return $db->getConnection()->lastInsertId();
}

/**
 * Get pending unlock requests
 */
function get_pending_requests() {
    $sql = "SELECT ur.*, u.username, u.first_name, u.last_name, f.title, f.public_id 
            FROM unlock_requests ur 
            JOIN users u ON ur.user_id = u.user_id 
            JOIN files f ON ur.file_id = f.id 
            WHERE ur.status = 'pending' 
            ORDER BY ur.created_at DESC";
    return db_query($sql)->fetchAll();
}

/**
 * Update unlock request status
 */
function update_unlock_request($request_id, $status) {
    $sql = "UPDATE unlock_requests SET status = ?, updated_at = NOW() WHERE id = ?";
    db_query($sql, [$status, $request_id]);
    
    return true;
}

/**
 * Save uploaded file info to database
 */
function save_file_info($title, $filename, $public_id, $required_channels, $is_paid, $price_text) {
    $sql = "INSERT INTO files (title, filename, public_id, required_channels, is_paid, price_text, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())";
    db_query($sql, [$title, $filename, $public_id, $required_channels, $is_paid ? 1 : 0, $price_text]);
    
    return $db->getConnection()->lastInsertId();
}

/**
 * Delete file
 */
function delete_file($file_id) {
    $file = get_file_by_id($file_id);
    if ($file) {
        // Delete physical file
        $file_path = __DIR__ . '/../uploads/' . $file['filename'];
        if (file_exists($file_path)) {
            unlink($file_path);
        }
        
        // Delete from database
        $sql = "DELETE FROM files WHERE id = ?";
        db_query($sql, [$file_id]);
        
        return true;
    }
    return false;
}
?>