<?php
// webhook.php - Main Bot Webhook

require_once 'config.php';
require_once 'inc/db.php';
require_once 'inc/functions.php';
require_once 'inc/telegram.php';

// Get input data
$input = file_get_contents('php://input');
$update = json_decode($input, true);

if (!$update) {
    exit;
}

// Process update
try {
    if (isset($update['message'])) {
        process_message($update['message']);
    } elseif (isset($update['callback_query'])) {
        process_callback_query($update['callback_query']);
    }
} catch (Exception $e) {
    error_log("Error processing update: " . $e->getMessage());
}

/**
 * Process incoming message
 */
function process_message($message) {
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $username = $message['from']['username'] ?? '';
    $first_name = $message['from']['first_name'] ?? '';
    $last_name = $message['from']['last_name'] ?? '';
    $text = $message['text'] ?? '';
    
    // Create or get user
    get_or_create_user($user_id, $username, $first_name, $last_name);
    
    log_activity($user_id, 'message_received', "Text: $text");
    
    if ($text == '/start') {
        show_main_menu($chat_id, $user_id);
    } elseif (strpos($text, '/fil') === 0) {
        $public_id = substr($text, 4);
        process_file_request($chat_id, $user_id, $public_id);
    } else {
        send_message($chat_id, "❌ Unknown command. Use /start to see available files.");
    }
}

/**
 * Process callback query
 */
function process_callback_query($callback_query) {
    $chat_id = $callback_query['message']['chat']['id'];
    $user_id = $callback_query['from']['id'];
    $callback_id = $callback_query['id'];
    $data = $callback_query['data'];
    $message_id = $callback_query['message']['message_id'];
    
    if ($data == 'check_membership') {
        // Get the original message to find which file was requested
        $message_text = $callback_query['message']['text'];
        
        // Extract public_id from message (this is a simplified approach)
        // In a real implementation, you might want to store this in callback data
        preg_match('/File: (.+?) -/', $message_text, $matches);
        if (isset($matches[1])) {
            $public_id = trim($matches[1]);
            process_file_request($chat_id, $user_id, $public_id, $message_id);
        } else {
            answer_callback_query($callback_id, "Please use the /fil command again.");
        }
    }
    
    answer_callback_query($callback_id, "");
}

/**
 * Show main menu with available files
 */
function show_main_menu($chat_id, $user_id) {
    $files = get_all_files();
    
    if (empty($files)) {
        send_message($chat_id, "📭 No files available at the moment.");
        return;
    }
    
    $message = "📁 <b>Available Files</b>\n\n";
    
    foreach ($files as $file) {
        $price_text = $file['is_paid'] ? "💰 Paid: " . $file['price_text'] : "🆓 Free";
        $message .= "/fil{$file['public_id']} - <b>{$file['title']}</b> - ({$price_text})\n\n";
    }
    
    $message .= "👉 Click on any file to unlock it!";
    
    send_message($chat_id, $message);
}

/**
 * Process file unlock request
 */
function process_file_request($chat_id, $user_id, $public_id, $edit_message_id = null) {
    $file = get_file_by_public_id($public_id);
    
    if (!$file) {
        $message = "❌ File not found.";
        if ($edit_message_id) {
            edit_message_text($chat_id, $edit_message_id, $message);
        } else {
            send_message($chat_id, $message);
        }
        return;
    }
    
    // Check channel membership
    $membership_check = check_channel_membership($user_id, $file['required_channels']);
    
    if ($membership_check !== true) {
        // User hasn't joined all channels
        $message = "🔒 <b>{$file['title']}</b>\n\n";
        $message .= "📋 <b>Requirements:</b>\n";
        $message .= "Please join the following channel(s) to access this file:\n\n";
        
        foreach ($membership_check as $channel) {
            $message .= "• @$channel\n";
        }
        
        $message .= "\nAfter joining, click the button below to verify.";
        
        $keyboard = create_join_keyboard($membership_check);
        
        if ($edit_message_id) {
            edit_message_text($chat_id, $edit_message_id, $message, $keyboard);
        } else {
            send_message($chat_id, $message, $keyboard);
        }
        
        log_activity($user_id, 'membership_required', "File: {$file['title']}");
        return;
    }
    
    // User has joined all required channels
    if (!$file['is_paid']) {
        // Free file - send immediately
        $file_path = __DIR__ . '/uploads/' . $file['filename'];
        
        if (file_exists($file_path)) {
            send_document($chat_id, $file_path, "📁 <b>{$file['title']}</b>\n\nEnjoy your file! 🎉");
            log_activity($user_id, 'file_sent_free', "File: {$file['title']}");
        } else {
            send_message($chat_id, "❌ File not found on server. Please contact admin.");
            log_activity($user_id, 'file_not_found', "File: {$file['title']}");
        }
    } else {
        // Paid file - create unlock request
        $request_id = create_unlock_request($user_id, $file['id']);
        
        $message = "💰 <b>Paid File: {$file['title']}</b>\n\n";
        $message .= "Price: <b>{$file['price_text']}</b>\n\n";
        $message .= "📨 Your unlock request has been sent to admin.\n";
        $message .= "You'll receive the file once approved.";
        
        if ($edit_message_id) {
            edit_message_text($chat_id, $edit_message_id, $message);
        } else {
            send_message($chat_id, $message);
        }
        
        // Notify admin
        $user_info = get_or_create_user($user_id, '', '', '');
        $admin_message = "🆕 <b>New Unlock Request</b>\n\n";
        $admin_message .= "👤 User: {$user_info['first_name']}";
        if ($user_info['username']) {
            $admin_message .= " (@{$user_info['username']})";
        }
        $admin_message .= "\n📁 File: {$file['title']}\n";
        $admin_message .= "💰 Price: {$file['price_text']}\n";
        $admin_message .= "🆔 Request ID: {$request_id}";
        
        send_message(ADMIN_ID, $admin_message);
        
        log_activity($user_id, 'unlock_request_created', "File: {$file['title']}, Request ID: {$request_id}");
    }
}

http_response_code(200);
?>