<?php
// admin.php - Admin Panel

require_once 'config.php';
require_once 'inc/db.php';
require_once 'inc/functions.php';
require_once 'inc/telegram.php';

session_start();

// Check admin authentication
function check_admin_auth() {
    if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        if (isset($_POST['admin_token'])) {
            if ($_POST['admin_token'] === ADMIN_TOKEN) {
                $_SESSION['admin_logged_in'] = true;
            } else {
                show_login_form("Invalid token!");
                exit;
            }
        } else {
            show_login_form();
            exit;
        }
    }
}

// Show login form
function show_login_form($error = '') {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Login - File Locker Bot</title>
        <style>
            body { font-family: Arial, sans-serif; max-width: 400px; margin: 100px auto; padding: 20px; }
            .login-form { background: #f9f9f9; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .form-group { margin-bottom: 15px; }
            label { display: block; margin-bottom: 5px; font-weight: bold; }
            input[type="password"] { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; }
            button { background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; }
            .error { color: red; margin-bottom: 15px; }
        </style>
    </head>
    <body>
        <div class="login-form">
            <h2>Admin Login</h2>
            <?php if ($error): ?>
                <div class="error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            <form method="post">
                <div class="form-group">
                    <label for="admin_token">Admin Token:</label>
                    <input type="password" id="admin_token" name="admin_token" required>
                </div>
                <button type="submit">Login</button>
            </form>
        </div>
    </body>
    </html>
    <?php
}

// Main admin panel
function show_admin_panel() {
    $action = $_GET['action'] ?? 'dashboard';
    
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Panel - File Locker Bot</title>
        <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; background: #f5f5f5; }
            .header { background: #007bff; color: white; padding: 15px 20px; }
            .nav { background: white; padding: 10px 20px; border-bottom: 1px solid #ddd; }
            .nav a { margin-right: 15px; text-decoration: none; color: #007bff; padding: 5px 10px; border-radius: 3px; }
            .nav a:hover, .nav a.active { background: #007bff; color: white; }
            .container { padding: 20px; }
            .card { background: white; padding: 20px; margin-bottom: 20px; border-radius: 5px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
            table { width: 100%; border-collapse: collapse; }
            th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }
            th { background: #f8f9fa; }
            .btn { padding: 8px 15px; border: none; border-radius: 3px; cursor: pointer; text-decoration: none; display: inline-block; }
            .btn-primary { background: #007bff; color: white; }
            .btn-success { background: #28a745; color: white; }
            .btn-danger { background: #dc3545; color: white; }
            .form-group { margin-bottom: 15px; }
            label { display: block; margin-bottom: 5px; font-weight: bold; }
            input, select, textarea { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 3px; }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>File Locker Bot - Admin Panel</h1>
        </div>
        
        <div class="nav">
            <a href="?action=dashboard" class="<?php echo $action == 'dashboard' ? 'active' : ''; ?>">Dashboard</a>
            <a href="?action=files" class="<?php echo $action == 'files' ? 'active' : ''; ?>">Files</a>
            <a href="?action=upload" class="<?php echo $action == 'upload' ? 'active' : ''; ?>">Upload File</a>
            <a href="?action=requests" class="<?php echo $action == 'requests' ? 'active' : ''; ?>">Unlock Requests</a>
            <a href="?action=logout" style="float: right;">Logout</a>
        </div>
        
        <div class="container">
            <?php
            switch ($action) {
                case 'dashboard':
                    show_dashboard();
                    break;
                case 'files':
                    show_files();
                    break;
                case 'upload':
                    show_upload_form();
                    break;
                case 'requests':
                    show_unlock_requests();
                    break;
                case 'approve':
                    approve_request($_GET['id']);
                    break;
                case 'reject':
                    reject_request($_GET['id']);
                    break;
                case 'delete':
                    delete_file($_GET['id']);
                    break;
                case 'logout':
                    session_destroy();
                    header('Location: admin.php');
                    exit;
                default:
                    show_dashboard();
            }
            ?>
        </div>
    </body>
    </html>
    <?php
}

// Dashboard
function show_dashboard() {
    $files_count = db_query("SELECT COUNT(*) as count FROM files")->fetch()['count'];
    $users_count = db_query("SELECT COUNT(*) as count FROM users")->fetch()['count'];
    $pending_requests = db_query("SELECT COUNT(*) as count FROM unlock_requests WHERE status = 'pending'")->fetch()['count'];
    $approved_requests = db_query("SELECT COUNT(*) as count FROM unlock_requests WHERE status = 'approved'")->fetch()['count'];
    ?>
    <div class="card">
        <h2>Dashboard</h2>
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-top: 20px;">
            <div style="background: #e3f2fd; padding: 20px; border-radius: 5px; text-align: center;">
                <h3>Total Files</h3>
                <p style="font-size: 24px; margin: 0;"><?php echo $files_count; ?></p>
            </div>
            <div style="background: #e8f5e8; padding: 20px; border-radius: 5px; text-align: center;">
                <h3>Total Users</h3>
                <p style="font-size: 24px; margin: 0;"><?php echo $users_count; ?></p>
            </div>
            <div style="background: #fff3cd; padding: 20px; border-radius: 5px; text-align: center;">
                <h3>Pending Requests</h3>
                <p style="font-size: 24px; margin: 0;"><?php echo $pending_requests; ?></p>
            </div>
            <div style="background: #d1ecf1; padding: 20px; border-radius: 5px; text-align: center;">
                <h3>Approved Requests</h3>
                <p style="font-size: 24px; margin: 0;"><?php echo $approved_requests; ?></p>
            </div>
        </div>
    </div>
    <?php
}

// Show files list
function show_files() {
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['delete_file'])) {
        delete_file($_POST['file_id']);
    }
    
    $files = get_all_files();
    ?>
    <div class="card">
        <h2>Uploaded Files</h2>
        <?php if (empty($files)): ?>
            <p>No files uploaded yet.</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Public ID</th>
                        <th>Type</th>
                        <th>Required Channels</th>
                        <th>Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($files as $file): ?>
                    <tr>
                        <td><?php echo $file['id']; ?></td>
                        <td><?php echo htmlspecialchars($file['title']); ?></td>
                        <td><?php echo $file['public_id']; ?></td>
                        <td><?php echo $file['is_paid'] ? '💰 Paid' : '🆓 Free'; ?></td>
                        <td><?php echo htmlspecialchars($file['required_channels']); ?></td>
                        <td><?php echo date('Y-m-d H:i', strtotime($file['created_at'])); ?></td>
                        <td>
                            <a href="?action=delete&id=<?php echo $file['id']; ?>" class="btn btn-danger" onclick="return confirm('Are you sure?')">Delete</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}

// Show upload form
function show_upload_form() {
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['file'])) {
        handle_file_upload();
    }
    ?>
    <div class="card">
        <h2>Upload New File</h2>
        <form method="post" enctype="multipart/form-data">
            <div class="form-group">
                <label for="title">File Title:</label>
                <input type="text" id="title" name="title" required>
            </div>
            
            <div class="form-group">
                <label for="file">Select File:</label>
                <input type="file" id="file" name="file" required accept="<?php echo implode(',', array_map(function($type) { return '.' . $type; }, ALLOWED_TYPES)); ?>">
                <small>Max size: <?php echo round(MAX_FILE_SIZE / 1024 / 1024); ?>MB</small>
            </div>
            
            <div class="form-group">
                <label for="public_id">Public ID (for /fil command):</label>
                <input type="text" id="public_id" name="public_id" required pattern="[a-zA-Z0-9_]+">
                <small>Only letters, numbers, and underscore. Example: premium_course</small>
            </div>
            
            <div class="form-group">
                <label for="required_channels">Required Channels (comma separated):</label>
                <input type="text" id="required_channels" name="required_channels" placeholder="@channel1, @channel2">
                <small>Without @, separate multiple channels with commas</small>
            </div>
            
            <div class="form-group">
                <label for="is_paid">
                    <input type="checkbox" id="is_paid" name="is_paid" value="1"> This is a paid file
                </label>
            </div>
            
            <div class="form-group" id="price_field" style="display: none;">
                <label for="price_text">Price Text:</label>
                <input type="text" id="price_text" name="price_text" placeholder="$10, 0.001 BTC, etc.">
            </div>
            
            <button type="submit" class="btn btn-primary">Upload File</button>
        </form>
    </div>
    
    <script>
        document.getElementById('is_paid').addEventListener('change', function() {
            document.getElementById('price_field').style.display = this.checked ? 'block' : 'none';
        });
    </script>
    <?php
}

// Handle file upload
function handle_file_upload() {
    $title = $_POST['title'];
    $public_id = $_POST['public_id'];
    $required_channels = $_POST['required_channels'];
    $is_paid = isset($_POST['is_paid']);
    $price_text = $_POST['price_text'] ?? '';
    
    // Check if public_id already exists
    $existing = get_file_by_public_id($public_id);
    if ($existing) {
        echo "<div style='color: red;'>Error: Public ID already exists!</div>";
        return;
    }
    
    $file = $_FILES['file'];
    
    // Validate file size
    if ($file['size'] > MAX_FILE_SIZE) {
        echo "<div style='color: red;'>Error: File too large!</div>";
        return;
    }
    
    // Validate file type
    $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($file_extension, ALLOWED_TYPES)) {
        echo "<div style='color: red;'>Error: File type not allowed!</div>";
        return;
    }
    
    // Generate unique filename
    $filename = uniqid() . '_' . $file['name'];
    $upload_path = __DIR__ . '/uploads/' . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $upload_path)) {
        // Save to database
        save_file_info($title, $filename, $public_id, $required_channels, $is_paid, $price_text);
        echo "<div style='color: green;'>File uploaded successfully!</div>";
    } else {
        echo "<div style='color: red;'>Error uploading file!</div>";
    }
}

// Show unlock requests
function show_unlock_requests() {
    $requests = get_pending_requests();
    ?>
    <div class="card">
        <h2>Pending Unlock Requests</h2>
        <?php if (empty($requests)): ?>
            <p>No pending requests.</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>User</th>
                        <th>File</th>
                        <th>Request Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($requests as $request): ?>
                    <tr>
                        <td><?php echo $request['id']; ?></td>
                        <td>
                            <?php echo htmlspecialchars($request['first_name'] . ' ' . $request['last_name']); ?>
                            <?php if ($request['username']): ?>
                                <br><small>@<?php echo $request['username']; ?></small>
                            <?php endif; ?>
                        </td>
                        <td><?php echo htmlspecialchars($request['title']); ?></td>
                        <td><?php echo date('Y-m-d H:i', strtotime($request['created_at'])); ?></td>
                        <td>
                            <a href="?action=approve&id=<?php echo $request['id']; ?>" class="btn btn-success">Approve</a>
                            <a href="?action=reject&id=<?php echo $request['id']; ?>" class="btn btn-danger">Reject</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}

// Approve unlock request
function approve_request($request_id) {
    $request = db_query("SELECT ur.*, u.user_id, f.title, f.filename FROM unlock_requests ur 
                        JOIN users u ON ur.user_id = u.user_id 
                        JOIN files f ON ur.file_id = f.id 
                        WHERE ur.id = ?", [$request_id])->fetch();
    
    if ($request) {
        // Update request status
        update_unlock_request($request_id, 'approved');
        
        // Send file to user
        $file_path = __DIR__ . '/uploads/' . $request['filename'];
        if (file_exists($file_path)) {
            send_document($request['user_id'], $file_path, "📁 <b>{$request['title']}</b>\n\nYour unlock request has been approved! 🎉");
            
            // Notify admin
            send_message(ADMIN_ID, "✅ Request #{$request_id} approved and file sent to user.");
            
            log_activity($request['user_id'], 'request_approved', "File: {$request['title']}");
        } else {
            send_message(ADMIN_ID, "❌ Error: File not found for request #{$request_id}");
        }
    }
    
    header('Location: admin.php?action=requests');
    exit;
}

// Reject unlock request
function reject_request($request_id) {
    $request = db_query("SELECT ur.*, u.user_id, f.title FROM unlock_requests ur 
                        JOIN users u ON ur.user_id = u.user_id 
                        JOIN files f ON ur.file_id = f.id 
                        WHERE ur.id = ?", [$request_id])->fetch();
    
    if ($request) {
        update_unlock_request($request_id, 'rejected');
        
        // Notify user
        send_message($request['user_id'], "❌ Your unlock request for '{$request['title']}' has been rejected by admin.");
        
        // Notify admin
        send_message(ADMIN_ID, "❌ Request #{$request_id} rejected.");
        
        log_activity($request['user_id'], 'request_rejected', "File: {$request['title']}");
    }
    
    header('Location: admin.php?action=requests');
    exit;
}

// Delete file
function delete_file($file_id) {
    if (delete_file($file_id)) {
        header('Location: admin.php?action=files&message=deleted');
    } else {
        header('Location: admin.php?action=files&message=error');
    }
    exit;
}

// Main execution
check_admin_auth();
show_admin_panel();
?>